#!/usr/bin/env python3

import sys
import json
import webbrowser
from PyQt5.QtCore import QTimer
from PyQt5.QtWidgets import (
    QApplication, QMainWindow, QWidget, QVBoxLayout, QHBoxLayout,
    QLabel, QComboBox, QScrollArea, QFrame, QPushButton, QTabWidget,
    QMessageBox, QTreeWidget, QTreeWidgetItem, QSplitter,
)
from PyQt5.QtCore import Qt, pyqtSignal
from PyQt5.QtGui import QFont, QCursor, QIcon


def carica_siti(file_json):
    try:
        with open(file_json, "r", encoding="utf-8") as f:
            return json.load(f)
    except Exception as e:
        print(f"Errore nel caricamento del file: {e}")
        return {}


def carica_preferiti():
    try:
        with open("preferiti.json", "r", encoding="utf-8") as f:
            return json.load(f)
    except:
        return []


def salva_preferiti(preferiti):
    try:
        with open("preferiti.json", "w", encoding="utf-8") as f:
            json.dump(preferiti, f, indent=2, ensure_ascii=False)
    except Exception as e:
        print(f"Errore nel salvataggio preferiti: {e}")


class SitoWidget(QWidget):
    aggiungi_preferito = pyqtSignal(dict, str)
    rimuovi_preferito = pyqtSignal(dict, str)
    
    def __init__(self, sito, paese, is_preferito=False):
        super().__init__()
        self.sito = sito
        self.paese = paese
        self.is_preferito = is_preferito
        self.setup_ui()
    
    def setup_ui(self):
        layout = QVBoxLayout(self)
        
        # Header con nome e pulsante preferito
        header_layout = QHBoxLayout()
        
        nome = QLabel(f"🔹 {self.sito['nome']}")
        nome.setFont(QFont("Arial", 11, QFont.Bold))
        header_layout.addWidget(nome)
        
        header_layout.addStretch()
        
        # Pulsante preferiti
        self.btn_preferito = QPushButton("⭐" if self.is_preferito else "☆")
        self.btn_preferito.setMaximumSize(30, 30)
        self.btn_preferito.setStyleSheet("""
            QPushButton {
                border: none;
                font-size: 16px;
                background: transparent;
            }
            QPushButton:hover {
                background: #e0e0e0;
                border-radius: 15px;
            }
        """)
        self.btn_preferito.clicked.connect(self.toggle_preferito)
        header_layout.addWidget(self.btn_preferito)
        
        layout.addLayout(header_layout)
        
        # Categoria
        if 'categoria' in self.sito:
            categoria_label = QLabel(f"Categoria: {self.sito['categoria']}")
            categoria_label.setFont(QFont("Arial", 9))
            categoria_label.setStyleSheet("color: #666; font-style: italic;")
            layout.addWidget(categoria_label)
        
        # Descrizione
        descrizione = QLabel(self.sito['descrizione'])
        descrizione.setWordWrap(True)
        layout.addWidget(descrizione)
        
        # URL
        url_label = QLabel(f"<a href='{self.sito['url']}'>{self.sito['url']}</a>")
        url_label.setOpenExternalLinks(True)
        url_label.setCursor(QCursor(Qt.PointingHandCursor))
        layout.addWidget(url_label)
        
        # Linea separatrice
        linea = QFrame()
        linea.setFrameShape(QFrame.HLine)
        linea.setFrameShadow(QFrame.Sunken)
        layout.addWidget(linea)
    
    def toggle_preferito(self):
        if self.is_preferito:
            self.rimuovi_preferito.emit(self.sito, self.paese)
            self.btn_preferito.setText("☆")
            self.is_preferito = False
        else:
            self.aggiungi_preferito.emit(self.sito, self.paese)
            self.btn_preferito.setText("⭐")
            self.is_preferito = True


class MainWindow(QMainWindow):
    def __init__(self, dati):
        super().__init__()
        self.setWindowTitle("Siti Governativi - Organizzati per Categoria")
        self.setGeometry(200, 100, 1000, 700)
        self.dati = dati
        self.preferiti = carica_preferiti()
        self.setup_ui()

    def setup_ui(self):
        # Widget principale
        main_widget = QWidget()
        self.setCentralWidget(main_widget)
        main_layout = QVBoxLayout(main_widget)
        
        # Dropdown per selezionare il paese
        paese_layout = QHBoxLayout()
        paese_layout.addWidget(QLabel("Seleziona una nazione:"))
        self.paesi_combo = QComboBox()
        self.paesi_combo.addItems(self.dati.keys())
        self.paesi_combo.currentTextChanged.connect(self.aggiorna_contenuto)
        paese_layout.addWidget(self.paesi_combo)
        paese_layout.addStretch()
        main_layout.addLayout(paese_layout)
        
        # Tab widget per organizzare per categoria e preferiti
        self.tab_widget = QTabWidget()
        main_layout.addWidget(self.tab_widget)
        
        # Tab delle categorie
        self.tab_categorie = QWidget()
        self.tab_widget.addTab(self.tab_categorie, "🗂️ Categorie")
        
        # Tab dei preferiti  
        self.tab_preferiti = QWidget()
        self.tab_widget.addTab(self.tab_preferiti, "⭐ Preferiti")
        
        self.setup_tab_categorie()
        self.setup_tab_preferiti()
        
        # Carica il contenuto iniziale
        self.aggiorna_contenuto(self.paesi_combo.currentText())

    def setup_tab_categorie(self):
        layout = QHBoxLayout(self.tab_categorie)
        
        # Splitter per dividere categorie e dettagli
        splitter = QSplitter(Qt.Horizontal)
        layout.addWidget(splitter)
        
        # Tree widget per le categorie
        self.tree_categorie = QTreeWidget()
        self.tree_categorie.setHeaderLabel("Categorie")
        self.tree_categorie.setMaximumWidth(300)
        self.tree_categorie.itemClicked.connect(self.categoria_selezionata)
        splitter.addWidget(self.tree_categorie)
        
        # Area di scorrimento per i siti della categoria selezionata
        self.scroll_area_categorie = QScrollArea()
        self.scroll_area_categorie.setWidgetResizable(True)
        self.scroll_content_categorie = QWidget()
        self.scroll_layout_categorie = QVBoxLayout(self.scroll_content_categorie)
        self.scroll_area_categorie.setWidget(self.scroll_content_categorie)
        splitter.addWidget(self.scroll_area_categorie)
        
        # Imposta le proporzioni del splitter
        splitter.setSizes([300, 700])

    def setup_tab_preferiti(self):
        layout = QVBoxLayout(self.tab_preferiti)
        
        # Area di scorrimento per i preferiti
        self.scroll_area_preferiti = QScrollArea()
        self.scroll_area_preferiti.setWidgetResizable(True)
        self.scroll_content_preferiti = QWidget()
        self.scroll_layout_preferiti = QVBoxLayout(self.scroll_content_preferiti)
        self.scroll_area_preferiti.setWidget(self.scroll_content_preferiti)
        layout.addWidget(self.scroll_area_preferiti)

    def categorizza_siti(self, siti):
        """Organizza i siti per categoria basandosi sul nome"""
        categorie = {
            "🏛️ Istituzioni Centrali": [],
            "👥 Parlamento": [],
            "⚖️ Giustizia": [],
            "🛡️ Sicurezza e Difesa": [],
            "💰 Finanze ed Economia": [],
            "🎓 Istruzione e Ricerca": [],
            "🏥 Salute": [],
            "🌱 Ambiente": [],
            "🏭 Industria e Lavoro": [],
            "🎨 Cultura e Turismo": [],
            "🚗 Trasporti": [],
            "🌾 Agricoltura": [],
            "💻 Digitale e Innovazione": [],
            "🔒 Controllo e Vigilanza": [],
            "👤 Servizi per il Cittadino": [],
            "📋 Altri Servizi": []
        }
        
        # Mappatura delle parole chiave alle categorie
        mappatura = {
            "presidente": "🏛️ Istituzioni Centrali",
            "premier": "🏛️ Istituzioni Centrali",
            "governo": "🏛️ Istituzioni Centrali",
            "presidenza": "🏛️ Istituzioni Centrali",
            "repubblica": "🏛️ Istituzioni Centrali",
            "elysee": "🏛️ Istituzioni Centrali",
            
            "senato": "👥 Parlamento",
            "camera": "👥 Parlamento",
            "deputati": "👥 Parlamento",
            "assemblée": "👥 Parlamento",
            "nationale": "👥 Parlamento",
            
            "giustizia": "⚖️ Giustizia",
            "costituzionale": "⚖️ Giustizia",
            "cassation": "⚖️ Giustizia",
            "conseil": "⚖️ Giustizia",
            "justice": "⚖️ Giustizia",
            
            "interno": "🛡️ Sicurezza e Difesa",
            "difesa": "🛡️ Sicurezza e Difesa",
            "intérieur": "🛡️ Sicurezza e Difesa",
            "armées": "🛡️ Sicurezza e Difesa",
            "defense": "🛡️ Sicurezza e Difesa",
            
            "economia": "💰 Finanze ed Economia",
            "finanze": "💰 Finanze ed Economia",
            "banca": "💰 Finanze ed Economia",
            "consob": "💰 Finanze ed Economia",
            "économie": "💰 Finanze ed Economia",
            "finances": "💰 Finanze ed Economia",
            "banque": "💰 Finanze ed Economia",
            "impots": "💰 Finanze ed Economia",
            
            "istruzione": "🎓 Istruzione e Ricerca",
            "università": "🎓 Istruzione e Ricerca",
            "ricerca": "🎓 Istruzione e Ricerca",
            "éducation": "🎓 Istruzione e Ricerca",
            
            "salute": "🏥 Salute",
            "santé": "🏥 Salute",
            "solidarités": "🏥 Salute",
            
            "ambiente": "🌱 Ambiente",
            "ecologica": "🌱 Ambiente",
            "écologique": "🌱 Ambiente",
            "ecologie": "🌱 Ambiente",
            
            "imprese": "🏭 Industria e Lavoro",
            "lavoro": "🏭 Industria e Lavoro",
            "made": "🏭 Industria e Lavoro",
            "travail": "🏭 Industria e Lavoro",
            
            "cultura": "🎨 Cultura e Turismo",
            "turismo": "🎨 Cultura e Turismo",
            "culture": "🎨 Cultura e Turismo",
            "beni": "🎨 Cultura e Turismo",
            
            "trasporti": "🚗 Trasporti",
            "infrastrutture": "🚗 Trasporti",
            
            "agricole": "🌾 Agricoltura",
            "agricoltura": "🌾 Agricoltura",
            "alimentari": "🌾 Agricoltura",
            "agriculture": "🌾 Agricoltura",
            
            "digitale": "💻 Digitale e Innovazione",
            "digitalizzazione": "💻 Digitale e Innovazione",
            "innovazione": "💻 Digitale e Innovazione",
            "agid": "💻 Digitale e Innovazione",
            "numérique": "💻 Digitale e Innovazione",
            
            "privacy": "🔒 Controllo e Vigilanza",
            "anticorruzione": "🔒 Controllo e Vigilanza",
            "anac": "🔒 Controllo e Vigilanza",
            "comptes": "🔒 Controllo e Vigilanza",
            
            "inps": "👤 Servizi per il Cittadino",
            "inail": "👤 Servizi per il Cittadino",
            "agenzia": "👤 Servizi per il Cittadino",
            "entrate": "👤 Servizi per il Cittadino",
            "poste": "👤 Servizi per il Cittadino",
            "anagrafe": "👤 Servizi per il Cittadino",
            "spid": "👤 Servizi per il Cittadino",
            "cie": "👤 Servizi per il Cittadino",
            "pagopa": "👤 Servizi per il Cittadino",
            "io.italia": "👤 Servizi per il Cittadino",
            "caf": "👤 Servizi per il Cittadino",
            "sociale": "👤 Servizi per il Cittadino",
            "pension": "👤 Servizi per il Cittadino",
            "retraite": "👤 Servizi per il Cittadino",
            "famille": "👤 Servizi per il Cittadino",
            "assurance": "👤 Servizi per il Cittadino",
            "maladie": "👤 Servizi per il Cittadino",
            "cpam": "👤 Servizi per il Cittadino",
            "cnam": "👤 Servizi per il Cittadino"
        }
        
        for sito in siti:
            nome_lower = sito['nome'].lower()
            url_lower = sito['url'].lower()
            categorizzato = False
            
            for keyword, categoria in mappatura.items():
                if keyword in nome_lower or keyword in url_lower:
                    categorie[categoria].append(sito)
                    categorizzato = True
                    break
            
            if not categorizzato:
                categorie["📋 Altri Servizi"].append(sito)
        
        # Rimuovi categorie vuote
        return {k: v for k, v in categorie.items() if v}

    def aggiorna_contenuto(self, paese):
        self.aggiorna_categorie(paese)
        self.aggiorna_preferiti()

    def aggiorna_categorie(self, paese):
        # Pulisce il tree widget
        self.tree_categorie.clear()
        
        # Pulisce l'area di visualizzazione
        self.pulisci_layout(self.scroll_layout_categorie)
        
        siti = self.dati.get(paese, [])
        if not siti:
            item = QTreeWidgetItem(["Nessun sito disponibile"])
            self.tree_categorie.addTopLevelItem(item)
            return
        
        # Categorizza i siti
        categorie = self.categorizza_siti(siti)
        
        # Popola il tree widget
        for categoria, siti_categoria in categorie.items():
            item = QTreeWidgetItem([f"{categoria} ({len(siti_categoria)})"])
            item.setData(0, Qt.UserRole, (categoria, siti_categoria, paese))
            self.tree_categorie.addTopLevelItem(item)
        
        # Seleziona la prima categoria
        if self.tree_categorie.topLevelItemCount() > 0:
            first_item = self.tree_categorie.topLevelItem(0)
            self.tree_categorie.setCurrentItem(first_item)
            self.categoria_selezionata(first_item, 0)

    def categoria_selezionata(self, item, column):
        data = item.data(0, Qt.UserRole)
        if not data:
            return
        
        categoria, siti_categoria, paese = data
        
        # Pulisce l'area di visualizzazione
        self.pulisci_layout(self.scroll_layout_categorie)
        
        # Aggiunge i siti della categoria
        for sito in siti_categoria:
            is_preferito = self.is_sito_preferito(sito, paese)
            sito_widget = SitoWidget(sito, paese, is_preferito)
            sito_widget.aggiungi_preferito.connect(self.aggiungi_a_preferiti)
            sito_widget.rimuovi_preferito.connect(self.rimuovi_da_preferiti)
            self.scroll_layout_categorie.addWidget(sito_widget)
        
        # Spacer per spingere tutto in alto
        self.scroll_layout_categorie.addStretch()

        # Scroll verso l'alto con ritardo 0 ms
        QTimer.singleShot(0, lambda: self.scroll_area_categorie.verticalScrollBar().setValue(2))
    
        # 🔧 Scrolla in cima
       # self.scroll_area_categorie.verticalScrollBar().setValue(0)

    def aggiorna_preferiti(self):
        # Pulisce l'area preferiti
        self.pulisci_layout(self.scroll_layout_preferiti)
        
        if not self.preferiti:
            label = QLabel("Nessun sito nei preferiti.\nAggiungi siti dai preferiti cliccando sulla stella ☆")
            label.setAlignment(Qt.AlignCenter)
            label.setStyleSheet("color: #666; font-style: italic; padding: 50px;")
            self.scroll_layout_preferiti.addWidget(label)
            return
        
        for preferito in self.preferiti:
            sito_widget = SitoWidget(preferito['sito'], preferito['paese'], True)
            sito_widget.aggiungi_preferito.connect(self.aggiungi_a_preferiti)
            sito_widget.rimuovi_preferito.connect(self.rimuovi_da_preferiti)
            self.scroll_layout_preferiti.addWidget(sito_widget)
        
        # Spacer per spingere tutto in alto
        self.scroll_layout_preferiti.addStretch()

    def is_sito_preferito(self, sito, paese):
        for preferito in self.preferiti:
            if (preferito['sito']['url'] == sito['url'] and 
                preferito['paese'] == paese):
                return True
        return False

    def aggiungi_a_preferiti(self, sito, paese):
        if not self.is_sito_preferito(sito, paese):
            self.preferiti.append({'sito': sito, 'paese': paese})
            salva_preferiti(self.preferiti)
            self.aggiorna_preferiti()

    def rimuovi_da_preferiti(self, sito, paese):
        self.preferiti = [p for p in self.preferiti 
                         if not (p['sito']['url'] == sito['url'] and p['paese'] == paese)]
        salva_preferiti(self.preferiti)
        self.aggiorna_preferiti()
        # Aggiorna anche la vista categorie se necessario
        if self.tab_widget.currentIndex() == 0:  # Tab categorie
            current_item = self.tree_categorie.currentItem()
            if current_item:
                self.categoria_selezionata(current_item, 0)

    def pulisci_layout(self, layout):
        while layout.count():
            item = layout.takeAt(0)
            widget = item.widget()
            if widget:
                widget.setParent(None)



if __name__ == "__main__":
    app = QApplication(sys.argv)
    dati_siti = carica_siti("siti_governativiV2.json")
    window = MainWindow(dati_siti)
    window.show()
    sys.exit(app.exec_())
